# JavaFX Settings Dialog with Theme Switching (FXML Version)

A comprehensive JavaFX application featuring a settings dialog with multiple tabs and dynamic theme switching, built using FXML for separation of concerns and better maintainability.

## Architecture

This application uses the **MVC (Model-View-Controller)** pattern with FXML:

- **Views (FXML files)**: Define the UI structure declaratively
  - `SettingsDialog.fxml` - Main settings dialog layout
  - `GeneralTab.fxml` - General settings tab
  - `ThemesTab.fxml` - Theme selection tab
  - `CertificatesTab.fxml` - Certificate management tab
  - `AboutTab.fxml` - About information tab

- **Controllers (Java classes)**: Handle user interactions and business logic
  - `SettingsController.java` - Main settings dialog controller
  - `ThemesTabController.java` - Themes tab specific controller
  - `ThemeManager.java` - Centralized theme management
  - `SettingsDialogApp.java` - Main application class

## Features

### 1. **General Tab**
   - Language selection dropdown
   - Auto-start on system startup option
   - Enable/disable notifications
   - Automatic update checks
   - Cache management
   - Reset to default settings

### 2. **Themes Tab**
   - **6 Beautiful Themes:**
     - Light (default white theme)
     - Dark (modern dark mode)
     - Ocean Blue (calming blue tones)
     - Nature Green (fresh green palette)
     - Royal Purple (elegant purple)
     - Sunset Red (warm red tones)
   - Interactive theme tiles with hover effects
   - Click any tile to instantly apply the theme
   - Visual color previews for each theme

### 3. **Certificates Tab**
   - Certificate list management
   - Import/Export certificates
   - Delete certificates
   - View certificate details
   - Trust settings for system and user certificates

### 4. **About Tab**
   - Application information
   - Version details
   - Copyright information
   - Quick links (Website, License, Support)
   - Technology information

## How to Compile and Run

### Prerequisites
- Java Development Kit (JDK) 11 or higher with JavaFX support
- JavaFX SDK (if not included with your JDK)

### File Structure
Ensure all files are in the same directory:
```
project/
├── SettingsDialogApp.java
├── SettingsController.java
├── ThemesTabController.java
├── ThemeManager.java
├── SettingsDialog.fxml
├── GeneralTab.fxml
├── ThemesTab.fxml
├── CertificatesTab.fxml
└── AboutTab.fxml
```

### Option 1: Using JavaFX SDK

1. **Download JavaFX SDK** from [https://gluonhq.com/products/javafx/](https://gluonhq.com/products/javafx/)

2. **Compile:**
   ```bash
   javac --module-path /path/to/javafx-sdk/lib --add-modules javafx.controls,javafx.fxml *.java
   ```

3. **Run:**
   ```bash
   java --module-path /path/to/javafx-sdk/lib --add-modules javafx.controls,javafx.fxml SettingsDialogApp
   ```

### Option 2: Using Maven

Create a `pom.xml` file:

```xml
<?xml version="1.0" encoding="UTF-8"?>
<project xmlns="http://maven.apache.org/POM/4.0.0"
         xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
         xsi:schemaLocation="http://maven.apache.org/POM/4.0.0 
         http://maven.apache.org/xsd/maven-4.0.0.xsd">
    <modelVersion>4.0.0</modelVersion>

    <groupId>com.example</groupId>
    <artifactId>settings-dialog</artifactId>
    <version>1.0.0</version>

    <properties>
        <maven.compiler.source>11</maven.compiler.source>
        <maven.compiler.target>11</maven.compiler.target>
        <javafx.version>17.0.2</javafx.version>
    </properties>

    <dependencies>
        <dependency>
            <groupId>org.openjfx</groupId>
            <artifactId>javafx-controls</artifactId>
            <version>${javafx.version}</version>
        </dependency>
        <dependency>
            <groupId>org.openjfx</groupId>
            <artifactId>javafx-fxml</artifactId>
            <version>${javafx.version}</version>
        </dependency>
    </dependencies>

    <build>
        <plugins>
            <plugin>
                <groupId>org.openjfx</groupId>
                <artifactId>javafx-maven-plugin</artifactId>
                <version>0.0.8</version>
                <configuration>
                    <mainClass>SettingsDialogApp</mainClass>
                </configuration>
            </plugin>
        </plugins>
    </build>
</project>
```

Then run:
```bash
mvn clean javafx:run
```

### Option 3: Using Gradle

Create a `build.gradle` file:

```gradle
plugins {
    id 'application'
    id 'org.openjfx.javafxplugin' version '0.0.13'
}

group 'com.example'
version '1.0.0'

repositories {
    mavenCentral()
}

javafx {
    version = "17.0.2"
    modules = ['javafx.controls', 'javafx.fxml']
}

application {
    mainClass = 'SettingsDialogApp'
}
```

Then run:
```bash
gradle run
```

## Usage

1. **Launch the application** - You'll see a main window with a "Open Settings" button
2. **Click "Open Settings"** to open the settings dialog
3. **Navigate between tabs** to explore different settings
4. **Click on any theme tile** in the Themes tab to instantly change the application's appearance
5. **Theme changes apply immediately** to both the main window and the settings dialog

## Theme Switching Details

The application supports real-time theme switching with the following themes:

| Theme | Primary Color | Accent Color | Best For |
|-------|--------------|--------------|----------|
| Light | White | Blue | Default, daytime use |
| Dark | Dark Gray | Purple | Night mode, reduced eye strain |
| Ocean Blue | Light Blue | Deep Blue | Calm, professional look |
| Nature Green | Light Green | Forest Green | Eco-friendly, nature themes |
| Royal Purple | Light Purple | Deep Purple | Creative, elegant designs |
| Sunset Red | Light Red | Deep Red | Bold, energetic interface |

Each theme tile shows a preview of its color scheme and applies instantly when clicked.

## Customization

You can easily extend this application by:

### Adding More Themes
1. Add theme colors in `ThemeManager.applyThemeToScene()` method
2. Create a new tile in `ThemesTabController.populateThemes()` or in the main app

### Adding More Settings
1. Edit the appropriate FXML file (e.g., `GeneralTab.fxml`)
2. Add corresponding `@FXML` fields in `SettingsController.java`
3. Implement the logic in controller methods

### Adding New Tabs
1. Create a new FXML file (e.g., `AdvancedTab.fxml`)
2. Add the tab in `SettingsDialog.fxml`:
   ```xml
   <Tab text="Advanced">
       <fx:include source="AdvancedTab.fxml"/>
   </Tab>
   ```
3. Add controller methods if needed

### Benefits of FXML Architecture
- **Separation of Concerns**: UI design separated from business logic
- **Easier Maintenance**: Modify UI without touching Java code
- **Better Collaboration**: Designers can work on FXML while developers work on controllers
- **Reusability**: FXML components can be easily reused across different parts of the application
- **SceneBuilder Support**: Can use JavaFX SceneBuilder for visual FXML editing

## Code Structure

### FXML Files
- `SettingsDialog.fxml` - Main dialog with TabPane and bottom buttons
- `GeneralTab.fxml` - General settings UI (language, notifications, etc.)
- `ThemesTab.fxml` - Themes grid container
- `CertificatesTab.fxml` - Certificate management UI
- `AboutTab.fxml` - About information and links

### Java Classes
- `SettingsDialogApp.java` - Main application entry point, creates main window
- `SettingsController.java` - Controller for main settings dialog and most tabs
- `ThemesTabController.java` - Specialized controller for theme management
- `ThemeManager.java` - Centralized theme management system
  - Manages multiple scenes
  - Applies themes dynamically
  - Provides theme data

### Key Methods
- `SettingsDialogApp.start()` - Initializes the main application window
- `SettingsDialogApp.showSettingsDialog()` - Loads and displays the FXML settings dialog
- `ThemeManager.applyTheme()` - Applies theme to all registered scenes
- `ThemeManager.registerScene()` - Registers a scene for theme management
- `SettingsController.handleApply()` - Saves and applies all settings
- `ThemesTabController.createThemeTile()` - Creates interactive theme tiles

## License

This is a demonstration project. Feel free to use and modify as needed.
