# FXML Settings Dialog - Quick Reference Guide

## Project Structure

```
settings-dialog/
├── SettingsDialogApp.java         # Main application class
├── SettingsController.java        # Main controller for settings dialog
├── ThemesTabController.java       # Controller for themes tab
├── ThemeManager.java              # Theme management system
├── SettingsDialog.fxml            # Main settings dialog layout
├── GeneralTab.fxml                # General settings tab
├── ThemesTab.fxml                 # Themes selection tab
├── CertificatesTab.fxml           # Certificates management tab
└── AboutTab.fxml                  # About information tab
```

## FXML to Controller Mapping

### SettingsDialog.fxml
**Controller:** `SettingsController`
- TabPane with 4 tabs (General, Themes, Certificates, About)
- Bottom button bar (Apply, OK, Cancel)
- Uses `<fx:include>` to load tab content

### GeneralTab.fxml
**Controller:** `SettingsController` (shared)
**Elements:**
- Language ComboBox (`languageComboBox`)
- Auto-start CheckBox (`autoStartCheckBox`)
- Notifications CheckBox (`notificationsCheckBox`)
- Auto-update CheckBox (`autoUpdateCheckBox`)
- Clear Cache Button (`clearCacheButton`)
- Reset Settings Button (`resetSettingsButton`)

### ThemesTab.fxml
**Controller:** `ThemesTabController`
**Elements:**
- GridPane for theme tiles (`themeGrid`)
- Theme tiles are added programmatically

### CertificatesTab.fxml
**Controller:** `SettingsController` (shared)
**Elements:**
- Certificate ListView (`certificateListView`)
- Import Button (`importButton`)
- Export Button (`exportButton`)
- Delete Button (`deleteButton`)
- View Details Button (`viewButton`)
- Trust System CheckBox (`trustSystemCheckBox`)
- Trust User CheckBox (`trustUserCheckBox`)

### AboutTab.fxml
**Controller:** `SettingsController` (shared)
**Elements:**
- Website Hyperlink (`websiteLink`)
- License Hyperlink (`licenseLink`)
- Support Hyperlink (`supportLink`)
- Tech Label (`techLabel`)

## Adding New Settings

### 1. Add to FXML
Edit the appropriate FXML file and add your control:

```xml
<CheckBox fx:id="myNewSettingCheckBox" 
          text="Enable My New Feature" 
          selected="false"/>
```

### 2. Add to Controller
Add the `@FXML` annotation in `SettingsController.java`:

```java
@FXML private CheckBox myNewSettingCheckBox;
```

### 3. Add Logic
Implement the functionality in the controller:

```java
private void applySettings() {
    // ... existing code ...
    boolean myNewSetting = myNewSettingCheckBox.isSelected();
    System.out.println("My new setting: " + myNewSetting);
}
```

## Adding New Themes

### Method 1: In ThemesTabController
Edit `populateThemes()` method:

```java
VBox customTheme = createThemeTile(
    "Custom Theme",     // Name
    "#E0F7FA",         // Primary color
    "#B2EBF2",         // Secondary color
    "#00ACC1"          // Accent color
);
themeGrid.add(customTheme, 0, 2); // Column 0, Row 2
```

### Method 2: In ThemeManager
Add a new case in `applyThemeToScene()`:

```java
case "custom_theme":
    backgroundColor = "#E0F7FA";
    textColor = "#006064";
    accentColor = "#00ACC1";
    buttonColor = "#B2EBF2";
    break;
```

## Event Handling

### FXML Event Handlers
Events are defined in FXML and handled in controller:

**FXML:**
```xml
<Button text="Click Me" onAction="#handleButtonClick"/>
```

**Controller:**
```java
@FXML
private void handleButtonClick() {
    // Handle the event
}
```

### Programmatic Event Handlers
Set in Java code:

```java
myButton.setOnAction(e -> {
    // Handle the event
});
```

## Theme Management

### Register a Scene for Theming
```java
Scene scene = new Scene(root);
themeManager.registerScene(scene);
```

### Apply a Theme
```java
themeManager.applyTheme("dark");  // or "light", "ocean_blue", etc.
```

### Unregister a Scene
```java
themeManager.unregisterScene(scene);
```

### Get Current Theme
```java
String current = themeManager.getCurrentTheme();
```

## Common Customizations

### Change Dialog Size
In `SettingsDialog.fxml`:
```xml
<BorderPane prefHeight="600.0" prefWidth="800.0">
```

### Change Tab Order
Reorder `<Tab>` elements in `SettingsDialog.fxml`:
```xml
<tabs>
    <Tab text="About">...</Tab>
    <Tab text="General">...</Tab>
    <Tab text="Themes">...</Tab>
    <Tab text="Certificates">...</Tab>
</tabs>
```

### Add Validation
In controller before applying settings:
```java
@FXML
private void handleApply() {
    if (validateSettings()) {
        applySettings();
        showNotification("Settings applied successfully");
    } else {
        showError("Validation Error", "Please check your settings");
    }
}

private boolean validateSettings() {
    // Add validation logic
    return true;
}
```

### Persist Settings
Use Java Preferences API:

```java
import java.util.prefs.Preferences;

private void saveSettings() {
    Preferences prefs = Preferences.userNodeForPackage(SettingsController.class);
    prefs.put("language", languageComboBox.getValue());
    prefs.putBoolean("autoStart", autoStartCheckBox.isSelected());
    prefs.put("theme", themeManager.getCurrentTheme());
}

private void loadSettings() {
    Preferences prefs = Preferences.userNodeForPackage(SettingsController.class);
    String language = prefs.get("language", "English");
    boolean autoStart = prefs.getBoolean("autoStart", false);
    String theme = prefs.get("theme", "light");
    
    languageComboBox.setValue(language);
    autoStartCheckBox.setSelected(autoStart);
    themeManager.applyTheme(theme);
}
```

## Debugging Tips

### 1. FXML Loading Issues
If FXML fails to load:
```java
try {
    FXMLLoader loader = new FXMLLoader(getClass().getResource("SettingsDialog.fxml"));
    Parent root = loader.load();
} catch (IOException e) {
    e.printStackTrace();  // Shows detailed error
}
```

### 2. Check fx:id Matches
Ensure FXML `fx:id` matches Java `@FXML` field name:
```xml
<Button fx:id="myButton" .../>
```
```java
@FXML private Button myButton;  // Names must match!
```

### 3. Verify Controller Path
In FXML, ensure controller class name is correct:
```xml
fx:controller="SettingsController"
```
If in a package:
```xml
fx:controller="com.example.SettingsController"
```

### 4. Test Scene Registration
Add logging to verify scenes are registered:
```java
public void registerScene(Scene scene) {
    System.out.println("Registering scene: " + scene);
    managedScenes.add(scene);
}
```

## Best Practices

1. **Keep FXML Simple** - Complex layouts should still be in FXML, but dynamic content in Java
2. **Use fx:include** - Break large FXML files into smaller, reusable components
3. **Consistent Naming** - Use descriptive IDs: `saveButton`, `userNameTextField`
4. **One Controller Per FXML** - Or use nested controllers for complex UIs
5. **Theme Centralization** - Keep all theme logic in `ThemeManager`
6. **Resource Cleanup** - Unregister scenes when dialogs close
7. **Error Handling** - Always handle FXML loading exceptions
8. **Comments** - Document complex FXML structures and controller logic

## Common Patterns

### Modal Dialog Pattern
```java
Stage dialog = new Stage();
dialog.initModality(Modality.APPLICATION_MODAL);
dialog.initOwner(primaryStage);
FXMLLoader loader = new FXMLLoader(getClass().getResource("MyDialog.fxml"));
Parent root = loader.load();
dialog.setScene(new Scene(root));
dialog.showAndWait();
```

### Passing Data to Dialog
```java
MyDialogController controller = loader.getController();
controller.setData(myData);
dialog.showAndWait();
String result = controller.getResult();
```

### Responsive Layout
Use layout constraints in FXML:
```xml
<VBox spacing="10.0" VBox.vgrow="ALWAYS">
    <Label text="Content" VBox.vgrow="ALWAYS"/>
</VBox>
```

## Resources

- JavaFX Documentation: https://openjfx.io/
- FXML Reference: https://openjfx.io/javadoc/17/javafx.fxml/javafx/fxml/doc-files/introduction_to_fxml.html
- SceneBuilder: https://gluonhq.com/products/scene-builder/
